<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class MenuManager
{
    protected $menuItems = [];
    protected $categories = [];

    public function __construct()
    {
        $this->loadMenuItems();
    }

    /**
     * Add a menu item to the admin panel
     */
    public function addMenuItem($categoryKey, $menuItem)
    {
        try {
            // Validate menu item structure
            $validatedItem = $this->validateMenuItem($menuItem);
            
            if (!$validatedItem) {
                return false;
            }

            // Initialize category if it doesn't exist
            if (!isset($this->menuItems[$categoryKey])) {
                $this->menuItems[$categoryKey] = [];
            }

            // Add the menu item
            $this->menuItems[$categoryKey][] = $validatedItem;
            
            // Save to cache
            $this->saveMenuItems();
            
            Log::info("Menu item added to category '{$categoryKey}': {$validatedItem['label']}");
            return true;

        } catch (\Exception $e) {
            Log::error("Failed to add menu item: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Add a custom category
     */
    public function addCategory($key, $label, $order = 100)
    {
        try {
            $this->categories[$key] = [
                'label' => $label,
                'order' => $order,
                'plugin_created' => true
            ];
            
            $this->saveMenuItems();
            
            Log::info("Menu category added: {$label}");
            return true;

        } catch (\Exception $e) {
            Log::error("Failed to add menu category: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Remove menu items by plugin name
     */
    public function removePluginMenuItems($pluginName)
    {
        try {
            $removed = false;
            
            foreach ($this->menuItems as $categoryKey => &$items) {
                $originalCount = count($items);
                $items = array_filter($items, function($item) use ($pluginName) {
                    return ($item['plugin'] ?? null) !== $pluginName;
                });
                
                // Re-index the array to avoid gaps
                $items = array_values($items);
                
                if (count($items) < $originalCount) {
                    $removed = true;
                }
            }

            foreach ($this->categories as $key => $category) {
                if (($category['plugin_created'] ?? false) && empty($this->menuItems[$key])) {
                    unset($this->categories[$key]);
                    unset($this->menuItems[$key]);
                }
            }
            
            if ($removed) {
                $this->saveMenuItems();
                Log::info("Removed menu items for plugin: {$pluginName}");
            }
            
            return $removed;

        } catch (\Exception $e) {
            Log::error("Failed to remove plugin menu items: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get all menu items organized by category
     */
    public function getMenuItems()
    {
        return $this->menuItems;
    }

    /**
     * Get menu categories
     */
    public function getCategories()
    {
        $defaultCategories = [
            'main' => ['label' => 'Main', 'order' => 10],
            'management' => ['label' => 'Management', 'order' => 20],
            'forum' => ['label' => 'Forum', 'order' => 30],
            'billing' => ['label' => 'Billing', 'order' => 40],
            'configuration' => ['label' => 'Configuration', 'order' => 50],
        ];

        $allCategories = array_merge($defaultCategories, $this->categories);
        
        uasort($allCategories, function($a, $b) {
            return ($a['order'] ?? 100) - ($b['order'] ?? 100);
        });
        
        return $allCategories;
    }

    /**
     * Get menu items for a specific category, filtered by active plugins
     */
    public function getCategoryItems($categoryKey)
    {
        $items = $this->menuItems[$categoryKey] ?? [];
        
        return array_filter($items, function($item) {
            if (!isset($item['plugin'])) {
                return true;
            }
            
            try {
                $pluginManager = app(\App\Services\PluginManager::class);
                return $pluginManager->isPluginActive($item['plugin']);
            } catch (\Exception $e) {
                Log::warning("Failed to check plugin status for menu item: " . $e->getMessage());
                return false;
            }
        });
    }

    /**
     * Check if user can access menu item
     */
    public function canAccessMenuItem($menuItem)
    {
        if (isset($menuItem['permission']) && !auth()->user()->can($menuItem['permission'])) {
            return false;
        }

        if (isset($menuItem['roles'])) {
            $userRoles = auth()->user()->roles->pluck('name')->toArray();
            $requiredRoles = is_array($menuItem['roles']) ? $menuItem['roles'] : [$menuItem['roles']];
            
            if (!array_intersect($userRoles, $requiredRoles)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Validate menu item structure
     */
    protected function validateMenuItem($menuItem)
    {
        $required = ['label', 'url', 'icon'];
        
        foreach ($required as $field) {
            if (!isset($menuItem[$field]) || empty($menuItem[$field])) {
                Log::warning("Menu item missing required field: {$field}");
                return false;
            }
        }

        // Set defaults
        return array_merge([
            'permission' => null,
            'roles' => null,
            'active_routes' => [],
            'badge' => null,
            'target' => '_self',
            'plugin' => null,
            'order' => 100,
        ], $menuItem);
    }

    /**
     * Load menu items from cache
     */
    protected function loadMenuItems()
    {
        $cached = Cache::get('admin_menu_items', []);
        $this->menuItems = $cached['items'] ?? [];
        $this->categories = $cached['categories'] ?? [];
    }

    /**
     * Save menu items to cache
     */
    protected function saveMenuItems()
    {
        Cache::put('admin_menu_items', [
            'items' => $this->menuItems,
            'categories' => $this->categories,
        ], 60 * 60 * 24); // Cache for 24 hours
    }

    /**
     * Clear all cached menu items
     */
    public function clearCache()
    {
        Cache::forget('admin_menu_items');
        $this->menuItems = [];
        $this->categories = [];
        
        Log::info("Admin menu cache cleared");
    }

    /**
     * Clean up orphaned menu items from inactive plugins
     */
    public function cleanupOrphanedItems()
    {
        try {
            $pluginManager = app(\App\Services\PluginManager::class);
            $activePlugins = $pluginManager->getActivePlugins();
            $cleaned = false;

            foreach ($this->menuItems as $categoryKey => &$items) {
                $originalCount = count($items);
                
                $items = array_filter($items, function($item) use ($activePlugins) {
                    if (!isset($item['plugin'])) {
                        return true;
                    }
                    
                    return in_array($item['plugin'], $activePlugins);
                });
                
                $items = array_values($items);
                
                if (count($items) < $originalCount) {
                    $cleaned = true;
                }
            }

            // Remove empty plugin-created categories
            $originalCategoryCount = count($this->categories);
            foreach ($this->categories as $key => $category) {
                if (($category['plugin_created'] ?? false) && empty($this->menuItems[$key])) {
                    unset($this->categories[$key]);
                    unset($this->menuItems[$key]);
                }
            }

            if ($cleaned || count($this->categories) < $originalCategoryCount) {
                $this->saveMenuItems();
                Log::info("Cleaned up orphaned menu items");
            }

            return $cleaned;

        } catch (\Exception $e) {
            Log::error("Failed to cleanup orphaned menu items: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Register a plugin's menu configuration
     */
    public function registerPluginMenu($pluginName, $menuConfig)
    {
        try {
            // Remove existing menu items for this plugin first to prevent duplicates
            $this->removePluginMenuItems($pluginName);

            if (isset($menuConfig['categories'])) {
                foreach ($menuConfig['categories'] as $key => $category) {
                    if (!isset($this->categories[$key])) {
                        $this->addCategory($key, $category['label'], $category['order'] ?? 100);
                    }
                }
            }

            if (isset($menuConfig['items'])) {
                foreach ($menuConfig['items'] as $categoryKey => $items) {
                    foreach ($items as $item) {
                        $item['plugin'] = $pluginName;
                        
                        $existingItems = $this->getCategoryItems($categoryKey);
                        $isDuplicate = false;
                        
                        foreach ($existingItems as $existingItem) {
                            if (($existingItem['plugin'] ?? '') === $pluginName && 
                                $existingItem['label'] === $item['label'] && 
                                $existingItem['url'] === $item['url']) {
                                $isDuplicate = true;
                                break;
                            }
                        }
                        
                        if (!$isDuplicate) {
                            $this->addMenuItem($categoryKey, $item);
                        }
                    }
                }
            }

            Log::info("Registered menu configuration for plugin: {$pluginName}");
            return true;

        } catch (\Exception $e) {
            Log::error("Failed to register plugin menu for {$pluginName}: " . $e->getMessage());
            return false;
        }
    }
}